
#include "drawutil.h"
#include <GL/gl.h>


void drawBox(
   const AlignedBox& box0, bool expand, bool isCool
) {
   AlignedBox box( box0 );

   // expand the box
   if ( expand ) {
      float diagonal = box.getDiagonal().length();
      diagonal /= 30;
      Vector3 v( diagonal, diagonal, diagonal );
      box = AlignedBox( box.getMin() - v, box.getMax() + v );
   }

   // now draw it.
   if ( ! isCool ) {
      glBegin( GL_LINE_STRIP );
         glVertex3fv( box.getCorner( 0 ).get() );
         glVertex3fv( box.getCorner( 1 ).get() );
         glVertex3fv( box.getCorner( 3 ).get() );
         glVertex3fv( box.getCorner( 2 ).get() );
         glVertex3fv( box.getCorner( 6 ).get() );
         glVertex3fv( box.getCorner( 7 ).get() );
         glVertex3fv( box.getCorner( 5 ).get() );
         glVertex3fv( box.getCorner( 4 ).get() );
         glVertex3fv( box.getCorner( 0 ).get() );
         glVertex3fv( box.getCorner( 2 ).get() );
      glEnd();
      glBegin( GL_LINES );
         glVertex3fv( box.getCorner( 1 ).get() );
         glVertex3fv( box.getCorner( 5 ).get() );
         glVertex3fv( box.getCorner( 3 ).get() );
         glVertex3fv( box.getCorner( 7 ).get() );
         glVertex3fv( box.getCorner( 4 ).get() );
         glVertex3fv( box.getCorner( 6 ).get() );
      glEnd();
   }
   else {
      glBegin( GL_LINES );
         for ( int dim = 0; dim < 3; ++dim ) {
            Vector3 v = ( box.getCorner(1<<dim) - box.getCorner(0) )*0.1;
            for ( int a = 0; a < 2; ++a ) {
               for ( int b = 0; b < 2; ++b ) {
                  int i = (a << ((dim+1)%3)) | (b << ((dim+2)%3));
                  glVertex3fv( box.getCorner(i).get() );
                  glVertex3fv( ( box.getCorner(i) + v ).get() );
                  i |= 1 << dim;
                  glVertex3fv( box.getCorner(i).get() );
                  glVertex3fv( ( box.getCorner(i) - v ).get() );
               }
            }
         }
      glEnd();
   }
}

void drawCircle(
   const Point3& centre,
   const Vector3& normal,
   float radius,
   float arcLengthPerPixel
) {
   Matrix m;
   m.setToLookAt( centre, centre+normal, normal.choosePerpendicular(), true );
   glPushMatrix();
      glMultMatrixf( m.get() );

      // We draw the circle as a polygon
      // with approximately one side per pixel.
      // The "+1" is to round up.
      int nbSides = (int)( 2 * M_PI * radius / arcLengthPerPixel + 1 );

      float deltaAngle = 2 * M_PI / nbSides;
      glBegin( GL_LINE_LOOP );
         for ( int i = 0; i < nbSides; ++i ) {
            float angle = i * deltaAngle;
            glVertex3f( radius * cos(angle), radius * sin(angle), 0 );
         }
      glEnd();
   glPopMatrix();
}

void drawCone(
   const Point3& apex,
   const Vector3& axis,
   float semiAngle,
   float lateralLength,
   int nbLateralSides,
   bool drawBase
) {
   Matrix m;
   m.setToLookAt( apex, apex+axis, axis.choosePerpendicular(), true );
   glPushMatrix();
      glMultMatrixf( m.get() );
      float h = lateralLength * cos( semiAngle );
      float r = lateralLength * sin( semiAngle );
      Point3 * basePoints = new Point3[ nbLateralSides ];
      float deltaAngle = 2*M_PI / nbLateralSides;
      int i;
      glBegin( GL_LINES );
         for ( i = 0; i < nbLateralSides; ++i ) {
            float angle = i * deltaAngle;
            basePoints[i] = Point3( r*cos(angle), r*sin(angle), -h );
            glVertex3f( 0, 0, 0 );
            glVertex3fv( basePoints[i].get() );
         }
      glEnd();
      if ( drawBase ) {
         glBegin( GL_LINE_LOOP );
            for ( i = 0; i < nbLateralSides; ++i ) {
               glVertex3fv( basePoints[i].get() );
            }
         glEnd();
      }
   glPopMatrix();
}

void drawFrame(
   const Point3& origin,
   const Vector3& i, const Vector3& j, const Vector3& k,
   float length, bool drawArrowHeads, bool isColoured,
   const Point3& i_colour,
   const Point3& j_colour,
   const Point3& k_colour
) {
   Point3 A = origin + (i*length);
   Point3 B = origin + (j*length);
   Point3 C = origin + (k*length);
   glBegin( GL_LINES );
      if ( isColoured ) glColor3fv( i_colour.get() );
      glVertex3fv( origin.get() );
      glVertex3fv( A.get() );
      if ( isColoured ) glColor3fv( j_colour.get() );
      glVertex3fv( origin.get() );
      glVertex3fv( B.get() );
      if ( isColoured ) glColor3fv( k_colour.get() );
      glVertex3fv( origin.get() );
      glVertex3fv( C.get() );
   glEnd();
   if ( drawArrowHeads ) {
      drawCone( C, -k, M_PI/8, length/7, 3, false );
      if ( isColoured ) glColor3fv( j_colour.get() );
      drawCone( B, -j, M_PI/8, length/7, 3, false );
      if ( isColoured ) glColor3fv( i_colour.get() );
      drawCone( A, -i, M_PI/8, length/7, 3, false );
   }
}

void drawCrossHairs( const Point3& centre, float radius ) {
   const float& r = radius;
   glBegin( GL_LINES );
      glVertex3fv( ( centre + Vector3(-r, 0, 0 ) ).get() );
      glVertex3fv( ( centre + Vector3( r, 0, 0 ) ).get() );
      glVertex3fv( ( centre + Vector3( 0,-r, 0 ) ).get() );
      glVertex3fv( ( centre + Vector3( 0, r, 0 ) ).get() );
      glVertex3fv( ( centre + Vector3( 0, 0,-r ) ).get() );
      glVertex3fv( ( centre + Vector3( 0, 0, r ) ).get() );
   glEnd();
}

void drawRay( const Ray& ray ) {
   const Point3& p = ray.origin;
   const Vector3& v = ray.direction;
   glBegin( GL_LINES );
      glVertex3fv( p.get() );
      glVertex3fv( ( p + v*5 ).get() );
   glEnd();
   drawCone( p+v*5, -v, M_PI/8, v.length()/3, 5, false );
}


