
#ifndef WORKSPACE_H
#define WORKSPACE_H


#include "Scene.h"


// This class is a kind of "glue" between the UI and the scene.
// It maintains the current selection, a display list for drawing the
// scene, and various modal flags to determine whether axes are drawn,
// or whether we're in "object selection" or "vertex selection" mode, etc.
//
// This class does *not* determine the mapping of mouse/keyboard events to
// function calls (that's supposed to happen in main.cc).
// It also does not provide a complete, one-stop API to clients;
// i.e. if a client wants to call a method on Scene or Object, it
// has to first retrieve the appropriate instance of Scene or Object
// and then make the call.  I did not indend for this class to become
// populated with one-liner inline methods that simply "forward" calls
// to Scene or Object.
class WorkSpace {
private:
   Scene _scene;
   list< OIndex > _selectedObjects;
   //list< VIndex > _selectedVertices; FIXME:vertex selection not yet supported
   Point3 _selectionCentre;

   bool _drawAxes;
   bool _drawManip;
   bool _drawGroundPlane;
   bool _vertexMode; // if false, we're in object mode

   //... _displayList; // FIXME: display lists not yet supported
   //bool _displayListDirty;

   void computeSelectionCentre();
public:

   WorkSpace() : _drawAxes( true ), _drawManip( true ) {}
   ~WorkSpace() {
      _selectedObjects.clear();
      // _selectedVertices.clear(); FIXME vertex selection not yet supported
   }

   void draw( Scene::DrawMode drawMode ) const;

   Scene& getScene() { return _scene; }

   // The returned list is unsorted.
   const list< OIndex >& getSelectedObjects() const { return _selectedObjects; }

   bool isSelected( OIndex objectIndex ) const {
      list< OIndex >::const_iterator it = _selectedObjects.begin();
      for ( ; it != _selectedObjects.end(); ++it )
         if ( *it == objectIndex ) return true;
      return false;
   }

   const Point3& getSelectionCentre() /* FIXME should be const */ {
      if ( _selectedObjects.empty() )
         _selectionCentre = Point3(0,0,0);
      return _selectionCentre;
   }

   void toggleDisplayOfAxes() { _drawAxes = ! _drawAxes; }

   void setSelection( const list< OIndex >& l ) {
      _selectedObjects = l;
      computeSelectionCentre();
   }
   // Any items passed in that are not already selected will be selected.
   // Any items passed in that are already selected will be unselected.
   // The list passed in need not be sorted.
   void xorSelection( list< OIndex >& l );

   void insertTetrahedron();
   void insertCube();
   void insertOctahedron();
   void insertFlatRectangularMesh( int rows, int cols );
   void insertSphere( int lats, int lons );
   // FIXME: not yet implemented
   // void insertDodecahedron();
   // void insertIcosahedron();
   // void insertTorus( int lat, int long );
   // void insertPyramid( int lat );
   // void insertPrism( int lat );
   // void insertSpring( ... );
   void insertSeashell();

   void translateSelection( const Vector3& v, bool onlyTransformLocalSpace );
   void rotateSelection(
      float angle_in_radians, const Vector3& axis,
      const Point3& rotationCentre, bool onlyTransformLocalSpace
   );
   void scaleSelection(
      const Vector3& v, const Point3& scaleCentre, bool onlyTransformLocalSpace
   );

   void deleteSelection();
};


#endif /* WORKSPACE_H */

