
#ifndef MANIPULATOR_H
#define MANIPULATOR_H


#include "Tool.h"
#include "mathutil.h"


class WorkSpace;


class Manipulator : public Tool {
private:
   WorkSpace & _workSpace;

   // If this is true, we perform transformations in terms of
   // the object's local space (e.g. we translate along
   // or rotate around the axes of local space).
   // Otherwise we transform in terms of world space.
   bool _useLocalSpace;

   // Normally, when we transform an object, it's local
   // space is carried with it (within world space).
   // For example, if we translate an object, the object
   // is moved through world space (whether the translation
   // is done in terms of local or world space).  As it is
   // moved, its local origin (and hence its local space)
   // is moved with it, so that its local origin will always
   // be located at the object's centre (for example).
   // However, when this flag is true, we do not transform
   // the object; we only transform its local space.
   // This allows the user to change the location of the
   // object's local origin (or "pivot-point", as it is sometimes
   // called), or to change the orientation of the object's
   // local frame.
   bool _onlyTransformLocalSpace;

   // If there's nothing selected, we're not visible.
   bool _visible;

   // Stores the space we're currently working in.
   Point3 _origin;
   Vector3 _frame[3]; // unit vectors

   // This scaling factor ensures that the manip retains a
   // constant screen size, even if the user zooms out.
   float _size;

   // Stores the pixel locations of the manip in the viewport.
   LineSegment _axis[3];

   // This is -1 for none, or one of 0,1,2 for x,y,z respectively.
   // When the mouse is not dragging, this indicates which
   // axis should be hightlighted because the mouse is over it.
   short _currentAxis;
   
   // Constants that can be tuned
   static const int _thresholdDistanceSquared;
   static const float _translationMultiplier;
   static const float _rotationMultiplier;
   static const float _scaleMultiplier;

   void computeCurrentAxis();

public:
   Manipulator( WorkSpace& ws ) :
      _workSpace( ws ),
      _useLocalSpace( false ), _onlyTransformLocalSpace( false ),
      _visible( false ),
      _size( 1 ),
      _currentAxis( -1 )
   { }
   virtual ~Manipulator() { }

   void toggleTransformationSpace() {
      _useLocalSpace = ! _useLocalSpace;
   }
   void toggleWhatsBeingTransformed() {
      _onlyTransformLocalSpace = ! _onlyTransformLocalSpace;
   }

   virtual Status press( int x, int y, bool LMB, bool MMB );
   virtual Status move( int x, int y );
   virtual Status release( int x, int y, bool LMB, bool MMB );

   virtual void draw();
};


#endif /* MANIPULATOR_H */

