
#ifndef CAMERA_H
#define CAMERA_H


#include "mathutil.h"


class Camera {
private:
   // window dimensions (measured between pixel *edges*)
   // (the only reason we need these is to convert client coordinates,
   // which are relative to the upper-left corner, to coordinates for
   // OpenGL, which are relative to the lower-left corner)
   int _window_width_in_pixels, _window_height_in_pixels;

   // pixel (centre) coordinates of the viewport's corners
   int _viewport_x1, _viewport_y1, _viewport_x2, _viewport_y2;

   // pixel coordinates of the viewport's centre.
   // These will be half-integers if the viewport's dimensions are even.
   float _viewport_cx, _viewport_cy;

   // viewport dimensions
   float _viewport_radius_in_pixels; // measured between pixel *centres*
   float _viewport_radius, _viewport_width, _viewport_height;

   // camera state
   Point3 _scene_centre;
   float _scene_radius, _minimum_feature_size;
   float _near_plane, _far_plane; // clipping planes
   Point3 _position; // point of view
   Point3 _target;   // point of interest
   Vector3 _up;      // a unit vector, perpendicular to the line-of-sight
   Vector3 _ground;  // a unit vector perpendicular to the ground plane
   Vector3 _initialGround;

   // constants that can be tuned
   static const float _initialFieldOfViewSemiAngle;
   static const float _fudgeFactor;
   static const float _nearPlaneFactor;
   static const float _farPlaneFactor;
   static const float _rotationSpeedInDegreesPerRadius; // for yaw, pitch, roll
   static const float _orbitingSpeedInDegreesPerRadius; // for orbit
   static const float _pushThreshold;

   void initialize(
      const Point3& scene_centre, float scene_radius,
      int window_width_in_pixels, int window_height_in_pixels,
      int viewport_x1, int viewport_y1, int viewport_x2, int viewport_y2
   );
   void computeViewportWidthAndHeight();

public:

   Camera(
      int window_width_in_pixels,  // measured between pixel edges
      int window_height_in_pixels, // measured between pixel edges
      float scene_radius,
      const Point3& scene_centre = Point3(0,0,0)
   ) {
      initialize(
         scene_centre, scene_radius,
         window_width_in_pixels, window_height_in_pixels,
         0, 0, window_width_in_pixels - 1, window_height_in_pixels - 1
      );
   }

   Camera(
      int window_width_in_pixels,  // measured between pixel edges
      int window_height_in_pixels, // measured between pixel edges
      int viewport_x1, int viewport_y1, // coordinates at pixel centre
      int viewport_x2, int viewport_y2, // coordinates at pixel centre
      float scene_radius,
      const Point3& scene_centre = Point3(0,0,0)
   ) {
      initialize(
         scene_centre, scene_radius,
         window_width_in_pixels, window_height_in_pixels,
         viewport_x1, viewport_y1, viewport_x2, viewport_y2
      );
   }

   void setSceneCentre( const Point3& p ) {
      _scene_centre = p;
   }

   // This causes the far plane to be recomputed.
   void setSceneRadius( float );

   // This causes the near plane to be recomputed.
   // Note that the argument passed in must be positive.
   // Also note that smaller arguments will pull the
   // near plane closer, but will also reduce the
   // z-buffer's precision.
   void setMinimumFeatureSize( float );

   void reset();

   void resizeViewport(
      int window_width_in_pixels, // measured between pixel edges
      int window_height_in_pixels // measured between pixel edges
   ) {
      resizeViewport(
         window_width_in_pixels, window_height_in_pixels,
         0, 0, window_width_in_pixels - 1, window_height_in_pixels - 1
      );
   }

   void resizeViewport(
      int window_width_in_pixels,  // measured between pixel edges
      int window_height_in_pixels, // measured between pixel edges
      int viewport_x1, int viewport_y1, // coordinates at pixel centre
      int viewport_x2, int viewport_y2  // coordinates at pixel centre
   );

   void transform();

   const Point3& getPosition() const { return _position; }
   const Point3& getTarget() const { return _target; }

   // Changes the field-of-view.
   void zoomIn( float delta_pixels );

   // These "orbit" the camera, causing the scene to "tumble".
   // Normally, orbiting is done around the camera's target.
   // However, if the client supplies a centre point, orbiting
   // will be done around the given centre, and the camera target
   // will be adjusted accordingly.
   void orbit(
      float old_x_pixels, float old_y_pixels,
      float new_x_pixels, float new_y_pixels
   );
   void orbit(
      float old_x_pixels, float old_y_pixels,
      float new_x_pixels, float new_y_pixels,
      const Point3& centre
   );
   void orbit( float delta_x_pixels, float delta_y_pixels ) {
      orbit(
         _viewport_cx, _viewport_cy,
         _viewport_cx + delta_x_pixels, _viewport_cy + delta_y_pixels
      );
   }
   void orbit(
      float delta_x_pixels, float delta_y_pixels,
      const Point3& centre
   ) {
      orbit(
         _viewport_cx, _viewport_cy,
         _viewport_cx + delta_x_pixels, _viewport_cy + delta_y_pixels,
         centre
      );
   }

   // Translates the point-of-interest in the camera plane.
   // Also translates the camera by a same amount.
   // Effectively *moves* the camera up/down/right/left,
   // without changing its orientation.
   // Some may refer to this as "panning" the camera.
   //
   void translateSceneRightAndUp( float delta_x_pixels, float delta_y_pixels );

   // Moves the camera forward or backward.  Some refer to
   // this as "tracking" the camera.
   // If the boolean flag is true, the target (or point of interest)
   // is moved with the camera, keeping the distance between the two
   // constant.
   void dollyCameraForward( float delta_pixels, bool pushTarget );

   // Changes the elevation angle of the camera.
   // Some refer to this as "tilting" the camera.
   void pitchCameraUp( float delta_pixels );

   // Changes the azimuth angle of the camera.
   // Some refer to this as "panning" the camera.
   void yawCameraRight( float delta_pixels );

   // Rolls the camera.
   void rollCameraRight( float delta_pixels );

   void lookAt( const Point3& );

   // Returns the ray through the centre of the given pixel.
   Ray computeRay( int pixel_x, int pixel_y ) const;
   // Computes the pixel covering the given point.
   // Also returns the z-distance (in camera space) to the point.
   float computePixel( const Point3&, int& pixel_x, int& pixel_y ) const;

   // Compute the necessary size, in world space, of an object
   // (at the given distance from the camera,
   // or centred at the given point, respectively)
   // for it to cover the given fraction of the viewport.
   // These methods are useful for determining the necessary size of widgets
   // placed in 3D space for them to have a constant size in screen space.
   float convertLength( float z_distance, float fractionOfViewportSize ) const;
   float convertLength( const Point3& p, float fractionOfViewportSize ) const {
      int x, y;
      return convertLength( computePixel( p, x, y ), fractionOfViewportSize );
   }

   // Compute the necessary size, in world space, of an object
   // centred at the given point
   // for it to cover the given length of pixels.
   float convertPixelLength( const Point3& p, int pixelLength ) const {
      int x, y;
      return convertLength(
         computePixel( p, x, y ),
         0.5 * pixelLength / (float)_viewport_radius_in_pixels
      );
   }
};


#endif /* CAMERA_H */

