//
// utils.cpp
//

#include <SDL_image.h>

#include "utils.h"


// get the smallest power of two greater than the input value
//
int
utils::powerOfTwo(int value)
{
	int result = 1;
	while( result < value )
		result <<= 1;
	return result;
}

// gets a surface pixel
// surface must be locked before calling this!
//
Uint32
utils::getPixel(SDL_Surface* surface, int x, int y)
{
	int bpp = surface->format->BytesPerPixel;
	
	// p is the address to the requested pixel
	//
	Uint8* p = (Uint8*)surface->pixels + y * surface->pitch + x * bpp;
	
	switch( bpp )
	{
		case 1:
			return *p;
		
		case 2:
			return *(Uint16*)p;
		
		case 3:
			if( SDL_BYTEORDER == SDL_BIG_ENDIAN )
				return p[0] << 16 | p[1] << 8 | p[2];
			else
				return p[0] | p[1] << 8 | p[2] << 16;
		
		case 4:
			return *(Uint32*)p;
		
		default:
			return 0;	// should never happen
	}
}

// sets a surface pixel
// surface must be locked before calling this!
//
void
utils::setPixel(SDL_Surface* surface, int x, int y, Uint32 pixel)
{
	int bpp = surface->format->BytesPerPixel;
	
	// p is the address to the requested pixel
	//
	Uint8* p = (Uint8*)surface->pixels + y * surface->pitch + x * bpp;
	
	switch( bpp )
	{
		case 1:
			*p = pixel;
			break;
		
		case 2:
			*(Uint16*)p = pixel;
			break;
		
		case 3:
			if( SDL_BYTEORDER == SDL_BIG_ENDIAN )
			{
				p[0] = (pixel >> 16) & 0xff;
				p[1] = (pixel >> 8) & 0xff;
				p[2] = pixel & 0xff;
			}
			else
			{
				p[0] = pixel & 0xff;
				p[1] = (pixel >> 8) & 0xff;
				p[2] = (pixel >> 16) & 0xff;
			}
			break;
		
		case 4:
			*(Uint32*)p = pixel;
			break;
	}
}

// flips an SDL surface according to the specified flags
// the returned surface must be freed by the caller!
// returns 0 on error
//
SDL_Surface*
utils::flipSurface(SDL_Surface* surface, int flags)
{
	// create blank surface based on supplied surface
	//
	SDL_Surface* flippedSurface = SDL_CreateRGBSurface(SDL_SWSURFACE,
													   surface->w,
													   surface->h,
													   surface->format->BitsPerPixel,
													   surface->format->Rmask,
													   surface->format->Gmask,
													   surface->format->Bmask,
													   surface->format->Amask);
	if( !flippedSurface )
		return 0;
	
	// lock the surface, if necessary
	//
	if( SDL_MUSTLOCK(surface) ) SDL_LockSurface(surface);
	
	// iterate through pixels and swap accordingly
	//
	Uint32 pixel;
	for( int x = 0, rx = flippedSurface->w - 1; x < flippedSurface->w; x++, rx-- )
	{
		for( int y = 0, ry = flippedSurface->h - 1; y < flippedSurface->h; y++, ry-- )
		{
			pixel = getPixel(surface, x, y);
			
			if( (flags & FLIP_VERT) && (flags & FLIP_HORZ) )
				setPixel(flippedSurface, rx, ry, pixel);
			else if( flags & FLIP_HORZ )
				setPixel(flippedSurface, rx, y, pixel);
			else if( flags & FLIP_VERT )
				setPixel(flippedSurface, x, ry, pixel);
			else	// no flip
				setPixel(flippedSurface, x, y, pixel);
		}
	}
	
	// unlock the surface, if necessary
	//
	if( SDL_MUSTLOCK(surface) ) SDL_UnlockSurface(surface);
	
	return flippedSurface;
}

// load image from file to SDL_Surface
// convert SDL_Surface to GL texture
// store texture coordinate range in texcoord:
//  (minx, miny, maxx, maxy)
// return texture object id (0 if error)
//
GLuint
utils::SDL_GL_LoadTextureFromFile(const std::string& filename, GLdouble* texcoord)
{
	// get image and make sure it loaded
	//
	SDL_Surface* surfaceOrig = IMG_Load(filename.c_str());
	if( !surfaceOrig )
		return 0;
	
	// SDL surfaces are inverted relative to OpenGL, so flip it
	//
	SDL_Surface* surface = flipSurface(surfaceOrig, FLIP_VERT);
	if( !surface )
	{
		SDL_FreeSurface(surfaceOrig);
		return 0;
	}
	
	// load texture from surface
	//
	GLuint texId = SDL_GL_LoadTexture(surface, texcoord);
	
	// free surfaces and return
	//
	SDL_FreeSurface(surfaceOrig);
	SDL_FreeSurface(surface);
	
	return texId;
}

GLuint
utils::SDL_GL_LoadTextureFromFile(const std::string& filename, Uint8 colorkeyRed, Uint8 colorkeyGreen, Uint8 colorkeyBlue, GLdouble* texcoord)
{
	// get image and make sure it loaded
	//
	SDL_Surface* surfaceOrig = IMG_Load(filename.c_str());
	if( !surfaceOrig )
		return 0;
	
	// SDL surfaces are inverted relative to OpenGL, so flip it
	//
	SDL_Surface* surface = flipSurface(surfaceOrig, FLIP_VERT);
	if( !surface )
	{
		SDL_FreeSurface(surfaceOrig);
		return 0;
	}
	
	// load texture from surface
	//
	GLuint texId = SDL_GL_LoadTexture(surface, colorkeyRed, colorkeyGreen, colorkeyBlue, texcoord);
	
	// free surfaces and return
	//
	SDL_FreeSurface(surfaceOrig);
	SDL_FreeSurface(surface);
	
	return texId;
}

GLuint
utils::SDL_GL_LoadTexture(SDL_Surface* surface, GLdouble* texcoord)
{
	// textures must be powers of 2, so find closest powers
	// of 2 to image dimensions
	// specify texture coordinates based on results
	//
	int w = powerOfTwo(surface->w);
	int h = powerOfTwo(surface->h);
	
	texcoord[0] = 0.0;
	texcoord[1] = 0.0;
	texcoord[2] = (GLdouble)surface->w / (GLdouble)w;
	texcoord[3] = (GLdouble)surface->h / (GLdouble)h;
	
	// create blank SDL surface that is GL texture format ready
	// the endian check is for specifying the RGBA masks
	//
	SDL_Surface* image = SDL_CreateRGBSurface(SDL_SWSURFACE,
											  w, h, 32,
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
											  0xff000000,
											  0x00ff0000,
											  0x0000ff00,
											  0x000000ff
#else
											  0x000000ff,
											  0x0000ff00,
											  0x00ff0000,
											  0xff000000
#endif
											  );
	if( !image )
		return 0;
	
	// save the alpha blending attributes
	//
	Uint32 saved_flags = surface->flags & (SDL_SRCALPHA | SDL_RLEACCELOK);
	Uint8  saved_alpha = surface->format->alpha;
	if( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA )
		SDL_SetAlpha(surface, 0, 0);
	
	// copy the surface into the GL texture image
	//
	SDL_Rect area;
	area.x = 0;
	area.y = 0;
	area.w = surface->w;
	area.h = surface->h;
	SDL_BlitSurface(surface, &area, image, &area);
	
	// restore the alpha blending attributes
	//
	if( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA )
		SDL_SetAlpha(surface, saved_flags, saved_alpha);
	
	// create an OpenGL texture for the image
	//
	GLuint texId;
	glGenTextures(1, &texId);
	glBindTexture(GL_TEXTURE_2D, texId);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexImage2D(GL_TEXTURE_2D,
				 0,
				 GL_RGBA,
				 w, h,
				 0,
				 GL_RGBA,
				 GL_UNSIGNED_BYTE,
				 image->pixels);
	
	// don't forget to free surfaces
	//
	SDL_FreeSurface(image);
	
	return texId;
}

GLuint
utils::SDL_GL_LoadTexture(SDL_Surface* surface, Uint8 colorkeyRed, Uint8 colorkeyGreen, Uint8 colorkeyBlue, GLdouble* texcoord)
{
	// textures must be powers of 2, so find closest powers
	// of 2 to image dimensions
	// specify texture coordinates based on results
	//
	int w = powerOfTwo(surface->w);
	int h = powerOfTwo(surface->h);
	
	texcoord[0] = 0.0;
	texcoord[1] = 0.0;
	texcoord[2] = (GLdouble)surface->w / (GLdouble)w;
	texcoord[3] = (GLdouble)surface->h / (GLdouble)h;
	
	// create blank SDL surface that is GL texture format ready
	// the endian check is for specifying the RGBA masks
	//
	SDL_Surface* image = SDL_CreateRGBSurface(SDL_SWSURFACE,
											  w, h, 32,
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
											  0xff000000,
											  0x00ff0000,
											  0x0000ff00,
											  0x000000ff
#else
											  0x000000ff,
											  0x0000ff00,
											  0x00ff0000,
											  0xff000000
#endif
											  );
	if( !image )
		return 0;
	
	// fill image with 0 alpha
	//
	Uint32 colorkey;
	colorkey = SDL_MapRGBA(image->format, 0x00, 0x00, 0x00, 0x00);
	SDL_FillRect(image, 0, colorkey);
	
	// set colorkey for surface
	//
	colorkey = SDL_MapRGBA(surface->format, colorkeyRed, colorkeyGreen, colorkeyBlue, 0x00);
	SDL_SetColorKey(surface, SDL_SRCCOLORKEY, colorkey);
	
	// copy the surface into the GL texture image
	//
	SDL_Rect area;
	area.x = 0;
	area.y = 0;
	area.w = surface->w;
	area.h = surface->h;
	SDL_BlitSurface(surface, &area, image, &area);
	
	// create an OpenGL texture for the image
	//
	GLuint texId;
	glGenTextures(1, &texId);
	glBindTexture(GL_TEXTURE_2D, texId);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexImage2D(GL_TEXTURE_2D,
				 0,
				 GL_RGBA,
				 w, h,
				 0,
				 GL_RGBA,
				 GL_UNSIGNED_BYTE,
				 image->pixels);
	
	// don't forget to free surfaces
	//
	SDL_FreeSurface(image);
	
	return texId;
}
