//
// main.cpp
//

#include <SDL.h>
#include <SDL_opengl.h>

#include <iostream>
#include <string>

#include "entity.h"
#include "timer.h"
#include "utils.h"


// constants
//
const int kDefWidth  = 640;
const int kDefHeight = 480;

const double kSecsPerFrame  = 1.0 / 60.0;	// 60 frames per second
const double kSecsPerUpdate = 1.0 / 90.0;	// 90 updates per second
const double kFPSReportInterval = 1.0;		// report every second

const GLdouble kFOVY     =  60.0;
const GLdouble kNearClip =   0.1;
const GLdouble kFarClip  = 100.0;


// globals
//
bool displayFPS = false;
bool exitGame   = false;

Cursor*    cursor;
Player*    player;
BulletList bullets;

GLdouble eyeZ = 6.5;

GLuint   bgTexId;
GLdouble bgTC[4];


// function declarations
//
bool setupGL(int argc, char** argv);
bool loadData();
void messageLoop();
void handleKeyboard(const SDL_KeyboardEvent& key);
void handleMouseButton(const SDL_MouseButtonEvent& button);
void handleMouseMotion(const SDL_MouseMotionEvent& motion);
void update(double dt);
void display();
void unloadData();
void quit(int code);


// main function
//
int
main(int argc, char** argv)
{
	// init SDL default, audio, video, and timer subsystems
	//
	if( SDL_Init(SDL_INIT_AUDIO | SDL_INIT_VIDEO | SDL_INIT_TIMER) == -1 )
	{
		std::cerr<< "Failed to initialize SDL: "<< SDL_GetError()<< std::endl;
		quit(1);
	}
	
	// init SDL OpenGL window
	//
	if( !setupGL(argc, argv) ) quit(2);
	
	// load game data
	//
	if( !loadData() ) quit(3);
	
	// start game loop
	//
	Timer updateTimer;
	Timer displayTimer;
	Timer fpsTimer;
	
	int frameCnt = 0;
	updateTimer.reset();
	displayTimer.reset();
	fpsTimer.reset();
	
	while( !exitGame )
	{
		// handle events
		//
		messageLoop();
		
		// update state
		//
		if( updateTimer.elapsed() > kSecsPerUpdate )
		{
			update(updateTimer.elapsed());
			updateTimer.reset();
		}
		
		// update display
		//
		if( displayTimer.elapsed() > kSecsPerFrame )
		{
			display();
			frameCnt++;
			displayTimer.reset();
		}
		
		// report FPS stats
		//
		if( displayFPS && fpsTimer.elapsed() > kFPSReportInterval )
		{
			std::cout<< "FPS: "<< double(frameCnt)/fpsTimer.elapsed()<< std::endl;
			
			frameCnt = 0;
			fpsTimer.reset();
		}
		
		// don't hog the CPU
		//
		SDL_Delay(1);
	}
	
	// unload game data
	//
	unloadData();
	
	// shutdown
	//
	quit(0);
	
	return 0;
}


//************************************************************


// sets up an SDL OpenGL window and inits GL states
//
bool
setupGL(int argc, char** argv)
{
	// get command line flags
	//
	bool verbose = false;
	int  flags   = SDL_OPENGL;
	
	for( int i = 1; i < argc; i++ )
	{
		std::string str(argv[i]);
		
		if( str == "-fullscreen" )
			flags |= SDL_FULLSCREEN;
		else if( str == "-verbose" )
			verbose = true;
	}
	
	// get video bits per pixel
	//
	const SDL_VideoInfo* vinfo = SDL_GetVideoInfo();
	if( !vinfo )
	{
		std::cerr<< "Failed to obtain video info: "<< SDL_GetError()<< std::endl;
		return false;
	}
	int bpp = vinfo->vfmt->BitsPerPixel;
	
	// specify OpenGL window attributes
	//
	SDL_GL_SetAttribute(SDL_GL_RED_SIZE,     8);
	SDL_GL_SetAttribute(SDL_GL_GREEN_SIZE,   8);
	SDL_GL_SetAttribute(SDL_GL_BLUE_SIZE,    8);
	SDL_GL_SetAttribute(SDL_GL_DEPTH_SIZE,   32);
	SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);
	
	// create window
	//
	SDL_WM_SetCaption("Ludum Dare 48", 0);
	SDL_WM_SetIcon(SDL_LoadBMP("gfx/icon.bmp"), 0);
	if( !SDL_SetVideoMode(kDefWidth, kDefHeight, bpp, flags) )
	{
		std::cerr<< "Failed to set video mode: "<< SDL_GetError()<< std::endl;
		return false;
	}
	SDL_ShowCursor(SDL_DISABLE);
	
	// setup GL viewport and projection
	//
	glViewport(0, 0, (GLsizei)kDefWidth, (GLsizei)kDefHeight);
	
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(kFOVY, (GLdouble)kDefWidth / (GLdouble)kDefHeight, kNearClip, kFarClip);
	
	glMatrixMode(GL_MODELVIEW);
	
	// initialize GL state **********
	//
	glClearColor(0.0, 0.0, 0.0, 0.0);
	glShadeModel(GL_SMOOTH);
	
	glEnable(GL_DEPTH_TEST);
	
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	
	glEnable(GL_TEXTURE_2D);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	
	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
	
	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	//
	//*******************************
	
	// report logistics, if necessary
	//
	if( verbose )
	{
		int attr;
		std::cout<< "bpp: "<< bpp<< std::endl;
		SDL_GL_GetAttribute(SDL_GL_RED_SIZE,     &attr);
		std::cout<< "SDL_GL_RED_SIZE    : "<< attr<< std::endl;
		SDL_GL_GetAttribute(SDL_GL_GREEN_SIZE,   &attr);
		std::cout<< "SDL_GL_GREEN_SIZE  : "<< attr<< std::endl;
		SDL_GL_GetAttribute(SDL_GL_BLUE_SIZE,    &attr);
		std::cout<< "SDL_GL_BLUE_SIZE   : "<< attr<< std::endl;
		SDL_GL_GetAttribute(SDL_GL_DEPTH_SIZE,   &attr);
		std::cout<< "SDL_GL_DEPTH_SIZE  : "<< attr<< std::endl;
		SDL_GL_GetAttribute(SDL_GL_DOUBLEBUFFER, &attr);
		std::cout<< "SDL_GL_DOUBLEBUFFER: "<< attr<< std::endl;
		std::cout<< std::endl;
		std::cout<< "OpenGL Vendor  : "<< glGetString(GL_VENDOR)<< std::endl;
		std::cout<< "OpenGL Renderer: "<< glGetString(GL_RENDERER)<< std::endl;
		std::cout<< "OpenGL Version : "<< glGetString(GL_VERSION)<< std::endl;
		std::cout<< std::endl;
		std::cout<< "GLU Version    : "<< gluGetString(GLU_VERSION)<< std::endl;
		std::cout<< std::endl;
	}
	
	return true;
}

// loads game data
//
bool
loadData()
{
	cursor = new Cursor;
	cursor->texId = utils::SDL_GL_LoadTextureFromFile("gfx/cursor.png", 0xff, 0xff, 0xff, cursor->texcoord);
	
	player = new Player;
	player->Entity::texId = utils::SDL_GL_LoadTextureFromFile("gfx/tankBody0.png", 0xff, 0xff, 0xff, player->Entity::texcoord);
	player->texId = utils::SDL_GL_LoadTextureFromFile("gfx/tankTurret.png", 0xff, 0xff, 0xff, player->texcoord);
	
	Bullet::texId = utils::SDL_GL_LoadTextureFromFile("gfx/bullet.png", 0x00, 0x00, 0x00, Bullet::texcoord);
	
	bgTexId = utils::SDL_GL_LoadTextureFromFile("gfx/background.png", bgTC);
	
	return true;
}

// processes events
//
void
messageLoop()
{
	SDL_Event event;
	
	while( SDL_PollEvent(&event) )
	{
		switch( event.type )
		{
			case SDL_KEYUP:
			case SDL_KEYDOWN:
				handleKeyboard(event.key);
				break;
			
			case SDL_MOUSEBUTTONUP:
			case SDL_MOUSEBUTTONDOWN:
				handleMouseButton(event.button);
				break;
			
			case SDL_MOUSEMOTION:
				handleMouseMotion(event.motion);
				break;
			
			case SDL_QUIT:
				exitGame = true;
				break;
		}
	}
}

// handle keyboard events
//
void
handleKeyboard(const SDL_KeyboardEvent& key)
{
	switch( key.keysym.sym )
	{
		case SDLK_w:
			if( key.state == SDL_PRESSED )
			{
				player->dy += 1.0;
				//std::cout<< "Up pressed"<< std::endl;
			}
			else	// SDL_RELEASED
			{
				player->dy -= 1.0;
				//std::cout<< "Up released"<< std::endl;
			}
			break;
		
		case SDLK_s:
			if( key.state == SDL_PRESSED )
			{
				player->dy -= 1.0;
				//std::cout<< "Down pressed"<< std::endl;
			}
			else	// SDL_RELEASED
			{
				player->dy += 1.0;
				//std::cout<< "Down released"<< std::endl;
			}
			break;
		
		case SDLK_a:
			if( key.state == SDL_PRESSED )
			{
				player->dx -= 1.0;
				//std::cout<< "Left pressed"<< std::endl;
			}
			else	// SDL_RELEASED
			{
				player->dx += 1.0;
				//std::cout<< "Left released"<< std::endl;
			}
			break;
		
		case SDLK_d:
			if( key.state == SDL_PRESSED )
			{
				player->dx += 1.0;
				//std::cout<< "Right pressed"<< std::endl;
			}
			else	// SDL_RELEASED
			{
				player->dx -= 1.0;
				//std::cout<< "Right released"<< std::endl;
			}
			break;
		
		case SDLK_EQUALS:
			if( key.state == SDL_PRESSED )
			{
				eyeZ -= 0.5;
				if( eyeZ < 0.0 ) eyeZ = 0.0;
				std::cout<< "eyeZ: "<< eyeZ<< std::endl;
			}
			break;
		
		case SDLK_MINUS:
			if( key.state == SDL_PRESSED )
			{
				eyeZ += 0.5;
				std::cout<< "eyeZ: "<< eyeZ<< std::endl;
			}
			break;
		
		case SDLK_SPACE:
			if( key.state == SDL_PRESSED )
			{
				std::cout<< "Space pressed"<< std::endl;
			}
			else	// SDL_RELEASED
			{
				std::cout<< "Space released"<< std::endl;
			}
			break;
		
		case SDLK_LCTRL:
		case SDLK_RCTRL:
			if( key.state == SDL_PRESSED )
			{
				std::cout<< "Ctrl pressed"<< std::endl;
			}
			else	// SDL_RELEASED
			{
				std::cout<< "Ctrl released"<< std::endl;
			}
			break;
		
		case SDLK_f:
			if( key.state == SDL_PRESSED )
				displayFPS = !displayFPS;
			break;
		
		case SDLK_ESCAPE:
			exitGame = true;
			break;
	}
}

// handle mouse button events
//
void
handleMouseButton(const SDL_MouseButtonEvent& button)
{
	switch( button.button )
	{
		// player shoots
		//
		case SDL_BUTTON_LEFT:
			if( button.state == SDL_PRESSED )
			{
				int x = button.x;
				int y = (kDefHeight - 1) - button.y;
				
				Bullet* bullet = player->shoot(x, y);
				bullets.push_back(bullet);
			}
			break;
	}
}

// handle mouse motion events
//
void
handleMouseMotion(const SDL_MouseMotionEvent& motion)
{
	// convert mouse coords to GL screen coord sys
	//  - SDL has (0,0) in upper left corner
	//  - OpenGL has (0,0) in lower left corner
	//
	int x = motion.x;
	int y = (kDefHeight - 1) - motion.y;
	
	// update cursor position
	//
	cursor->setPos(x, y);
	
	// update player's aim direction
	//
	player->aim(x, y);
}

// updates game state
//
void
update(double dt)
{
	// update entities
	//
	player->update(dt);
	
	BulletList::iterator bIter;
	for( bIter = bullets.begin(); bIter != bullets.end(); bIter++ )
		(*bIter)->update(dt);
	
	// remove all 'dead' entities
	//
	bIter = bullets.begin();
	while( bIter != bullets.end() )
	{
		if( (*bIter)->killme )
		{
			delete *bIter;
			bIter = bullets.erase(bIter);
		}
		else
			bIter++;
	}
}

// handles all display
//
void
display()
{
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	
	// position camera
	//
	glLoadIdentity();
	gluLookAt(player->x, player->y, eyeZ,
			  player->x, player->y, 0.0,
			  0.0, 1.0, 0.0);
	
	// draw background
	//
	glBindTexture(GL_TEXTURE_2D, bgTexId);
	glColor3d(1.0, 1.0, 1.0);
	glBegin(GL_QUADS);
		for( int x = -10; x < 20; x += 10 )
		{
			for( int y = -10; y < 20; y += 10 )
			{
				glTexCoord2d(bgTC[0], bgTC[1]); glVertex3d(x, y, -0.1);
				glTexCoord2d(bgTC[2], bgTC[1]); glVertex3d(x+10, y, -0.1);
				glTexCoord2d(bgTC[2], bgTC[3]); glVertex3d(x+10, y+10, -0.1);
				glTexCoord2d(bgTC[0], bgTC[3]); glVertex3d(x, y+10, -0.1);
			}
		}
	glEnd();
	
	// draw entities
	//
	player->display();
	
	BulletList::iterator bIter;
	for( bIter = bullets.begin(); bIter != bullets.end(); bIter++ )
		(*bIter)->display();
	
	// draw cursor
	//
	cursor->display();
	
	// flush GL and swap buffers
	//
	glFlush();
	SDL_GL_SwapBuffers();
}

// frees game data
//
void
unloadData()
{
	// delete all new'd items
	//
	delete cursor;
	delete player;
	
	BulletList::iterator bIter;
	for( bIter = bullets.begin(); bIter != bullets.end(); bIter++ )
		delete *bIter;
	
	// delete all texture ids
	//
	glDeleteTextures(1, &Bullet::texId);
	glDeleteTextures(1, &bgTexId);
}

// shuts down libraries and exits with specified code
//
void
quit(int code)
{
	SDL_Quit();
	exit(code);
}
