//
// entity.cpp
//

#include <cmath>

#include "entity.h"
#include "utils.h"


// Entity class **********
//
Entity::Entity() :
	killme(false), x(0.0), y(0.0), dx(0.0), dy(0.0),
	speed(0.0), r(1.0), g(1.0), b(1.0), texId(0),
	z(0.0), vx(0.0), vy(1.0)
{
	texcoord[0] = 0.0;
	texcoord[1] = 0.0;
	texcoord[2] = 0.0;
	texcoord[3] = 0.0;
}

Entity::~Entity()
{
	glDeleteTextures(1, &texId);
}

void
Entity::update(double dt)
{
	double norm = std::sqrt(dx * dx + dy * dy);
	if( norm >= utils::kEps )
	{
		vx = dx / norm;
		vy = dy / norm;
		
		x += dt * (speed * vx);
		y += dt * (speed * vy);
	}
}

void
Entity::display()
{
	glPushMatrix();
	glTranslated(x, y, z);
	
	// rotate (use [vx,vy] as y-axis)
	//
	GLdouble m[16] = {vy, -vx, 0, 0, vx, vy, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1};
	glMultMatrixd(m);
	
	glBindTexture(GL_TEXTURE_2D, texId);
	glColor3d(r, g, b);
	glBegin(GL_QUADS);
		glTexCoord2d(texcoord[0], texcoord[1]); glVertex3d(-0.5, -0.5, 0.0);
		glTexCoord2d(texcoord[2], texcoord[1]); glVertex3d( 0.5, -0.5, 0.0);
		glTexCoord2d(texcoord[2], texcoord[3]); glVertex3d( 0.5,  0.5, 0.0);
		glTexCoord2d(texcoord[0], texcoord[3]); glVertex3d(-0.5,  0.5, 0.0);
	glEnd();
	
	glPopMatrix();
}


// Cursor class **********
//
Cursor::Cursor() :
	sx(0), sy(0)
{
	z = 0.5;
}

Cursor::~Cursor()
{
}

void
Cursor::setPos(int scrnX, int scrnY)
{
	sx = scrnX;
	sy = scrnY;
}

void
Cursor::display()
{
	// update position, then display
	//
	GLint    viewport[4];
	GLdouble projMatrix[16];
	GLdouble mvMatrix[16];
	GLdouble worldZ;
	
	glGetIntegerv(GL_VIEWPORT, viewport);
	glGetDoublev(GL_PROJECTION_MATRIX, projMatrix);
	glGetDoublev(GL_MODELVIEW_MATRIX, mvMatrix);
	
	gluUnProject(sx, sy, utils::kCursorDepth, mvMatrix, projMatrix, viewport, &x, &y, &worldZ);
	
	Entity::display();
}


// Bullet class **********
//
GLuint   Bullet::texId       = 0;
GLdouble Bullet::texcoord[4] = {0.0, 0.0, 0.0, 0.0};

Bullet::Bullet() :
	dist(0.0)
{
	z     = 0.1;
	speed = 2.0;
}

Bullet::~Bullet()
{
	// don't let Entity destructor free texture
	//
	Entity::texId = 0;
}

void
Bullet::setTraj(double startX, double startY, double endX, double endY)
{
	double deltaX = endX - startX;
	double deltaY = endY - startY;
	
	double norm = std::sqrt(deltaX * deltaX + deltaY * deltaY);
	if( norm >= utils::kEps )
	{
		x    = startX;
		y    = startY;
		dx   = deltaX;
		dy   = deltaY;
		dist = norm;
	}
	else
		killme = true;
}

void
Bullet::update(double dt)
{
	// determine distance travelled this step
	//
	double startX = x;
	double startY = y;
	
	Entity::update(dt);
	
	double deltaX = x - startX;
	double deltaY = y - startY;
	double norm   = std::sqrt(deltaX * deltaX + deltaY * deltaY);
	
	// update distance and check for expiration
	//
	dist -= norm;
	if( dist <= 0.0 )
		killme = true;
}

void
Bullet::display()
{
	if( Entity::texId == 0 )
	{
		Entity::texId       = texId;
		Entity::texcoord[0] = texcoord[0];
		Entity::texcoord[1] = texcoord[1];
		Entity::texcoord[2] = texcoord[2];
		Entity::texcoord[3] = texcoord[3];
	}
	
	Entity::display();
}


// Player class **********
//
Player::Player() :
	texId(0), turretZ(0.01), ax(0.0), ay(1.0)
{
	texcoord[0] = 0.0;
	texcoord[1] = 0.0;
	texcoord[2] = 0.0;
	texcoord[3] = 0.0;
	
	speed = 1.0;
}

Player::~Player()
{
	glDeleteTextures(1, &texId);
}

// convert entity x,y position to screen space
// construct unit direction vector with arguments
// set as aim vector
//
void
Player::aim(int scrnX, int scrnY)
{
	GLint    viewport[4];
	GLdouble projMatrix[16];
	GLdouble mvMatrix[16];
	GLdouble winX, winY, winZ;
	
	// get all matrices
	//
	glGetIntegerv(GL_VIEWPORT, viewport);
	glGetDoublev(GL_PROJECTION_MATRIX, projMatrix);
	glGetDoublev(GL_MODELVIEW_MATRIX, mvMatrix);
	
	// project through pipeline
	//
	gluProject(x, y, turretZ, mvMatrix, projMatrix, viewport, &winX, &winY, &winZ);
	
	// construct aim vector
	//
	double tempX = scrnX - winX;
	double tempY = scrnY - winY;
	
	double norm = std::sqrt(tempX * tempX + tempY * tempY);
	if( norm >= utils::kEps )
	{
		ax = tempX / norm;
		ay = tempY / norm;
	}
}

Bullet*
Player::shoot(int scrnX, int scrnY)
{
	// get world space destination
	//
	GLint    viewport[4];
	GLdouble projMatrix[16];
	GLdouble mvMatrix[16];
	GLdouble worldX, worldY, worldZ;
	
	glGetIntegerv(GL_VIEWPORT, viewport);
	glGetDoublev(GL_PROJECTION_MATRIX, projMatrix);
	glGetDoublev(GL_MODELVIEW_MATRIX, mvMatrix);
	
	gluUnProject(scrnX, scrnY, utils::kCursorDepth, mvMatrix, projMatrix, viewport, &worldX, &worldY, &worldZ);
	
	// generate new bullet, set its trajectory and return it
	//
	Bullet* bullet = new Bullet;
	bullet->setTraj(x, y, worldX, worldY);
	return bullet;
}

void
Player::update(double dt)
{
	Entity::update(dt);
}

void
Player::display()
{
	Entity::display();
	
	glPushMatrix();
	glTranslated(x, y, turretZ);
	
	// rotate (use [ax,ay] as y-axis)
	//
	GLdouble m[16] = {ay, -ax, 0, 0, ax, ay, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1};
	glMultMatrixd(m);
	
	glBindTexture(GL_TEXTURE_2D, texId);
	glColor3d(r, g, b);
	glBegin(GL_QUADS);
		glTexCoord2d(texcoord[0], texcoord[1]); glVertex3d(-0.5, -0.5, 0.0);
		glTexCoord2d(texcoord[2], texcoord[1]); glVertex3d( 0.5, -0.5, 0.0);
		glTexCoord2d(texcoord[2], texcoord[3]); glVertex3d( 0.5,  0.5, 0.0);
		glTexCoord2d(texcoord[0], texcoord[3]); glVertex3d(-0.5,  0.5, 0.0);
	glEnd();
	
	glPopMatrix();
}
