#include "resource.h"
#include "engine.h"

#define MENU_HEIGHT 46
#define MENU_WIDTH 8
#define WINDOW_WIDTH (MENU_WIDTH) + (BUFF_WIDTH)
#define WINDOW_HEIGHT (MENU_HEIGHT) + (BUFF_HEIGHT)

//Globals
HDC g_mainDC = NULL;
HDC g_backDC = NULL;
HBITMAP g_back_bmp = NULL;
pPixel g_back_buff = NULL;
unsigned long* g_zBuff = NULL;

Triangle *g_tTri = NULL;
Mesh *g_mesh = NULL;
Camera *g_camera = NULL;

LRESULT CALLBACK WndProc(HWND, UINT, WPARAM, LPARAM);
void initGlobals(HWND);
void cleanup(HWND);
void render();

int WINAPI WinMain(HINSTANCE hInst, HINSTANCE hPrev, LPSTR lpCmdLine, int nShowCmd) {
	//Register the window class
	LPCSTR className = "ClassName";
	
	WNDCLASSEX wc;
	wc.cbSize = sizeof(WNDCLASSEX);
	wc.style = CS_HREDRAW | CS_VREDRAW;
	wc.lpfnWndProc = WndProc;
	wc.cbClsExtra = 0;
	wc.cbWndExtra = 0;
	wc.hInstance = hInst;
	wc.hIcon = LoadIcon(NULL, IDI_APPLICATION);
	wc.hCursor = LoadCursor(NULL, IDC_ARROW);
	wc.hbrBackground = (HBRUSH)(COLOR_WINDOW);
	wc.lpszMenuName = MAKEINTRESOURCE(IDR_MAINMENU);
	wc.lpszClassName = className;
	wc.hIconSm = LoadIcon(NULL, IDI_WINLOGO);
	
	if(!RegisterClassEx(&wc))
		return 1;
	
	//Create the window
	HWND hWnd = CreateWindowEx(0, className, "GDI Rendering Engine", WS_OVERLAPPEDWINDOW, CW_USEDEFAULT, CW_USEDEFAULT, WINDOW_WIDTH, WINDOW_HEIGHT, NULL, NULL, hInst, NULL);
	if(!hWnd)
		return 1;
	
	//Global setup
	initGlobals(hWnd);
	
	ShowWindow(hWnd, nShowCmd);
	UpdateWindow(hWnd);
	
	//Message loop
	bool done = false;
	MSG msg;
	LARGE_INTEGER time, freq;
	QueryPerformanceFrequency(&freq);
	freq.QuadPart /= 3; // /= PI (approx)
	Vector origin = Vector(0,0,0);
	while(!done) {
		if(PeekMessage(&msg, NULL, 0, 0, PM_REMOVE)) {
			if(msg.message == WM_QUIT)
				done = true;
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		} else {
			QueryPerformanceCounter(&time);
			double t = ((double)time.QuadPart)/freq.QuadPart;
			g_camera->pos = Vector(600*cos(t), 100, 600*sin(t));
			g_camera->transform = createTransform(origin - g_camera->pos, Vector(0, 1, 0));
			render();
		}
	}
	
	cleanup(hWnd);
	
	return (int)msg.wParam;
}

void initGlobals(HWND hWnd) {
	//Create a memory DC for back buffering
	g_mainDC = GetDC(hWnd);
	g_backDC = CreateCompatibleDC(g_mainDC);
	g_back_bmp = CreateCompatibleBitmap(g_mainDC, BUFF_WIDTH, BUFF_HEIGHT);
	SelectObject(g_backDC, g_back_bmp);
	g_back_buff = (pPixel)malloc(sizeof(Pixel) * BUFF_WIDTH * BUFF_HEIGHT);
	g_zBuff = (unsigned long*)malloc(sizeof(unsigned long) * BUFF_WIDTH * BUFF_HEIGHT);
	//Create a test triangle to draw
	g_mesh = new Mesh(2);
	g_mesh->add(new GouraudTriangle(Vector(100, -50, 50), Vector(0, 100, 0), Vector(-50, 0, 0), Vector(0,1,1), Vector(-1,0,1), Vector(1,-1,1), Pixel(0x00FF0000)));
	g_mesh->add(new GouraudTriangle(Vector(-100, 50, -50), Vector(0, -100, 0), Vector(50, 0, 0), Vector(0,0,1), Vector(0,0,1), Vector(0,0,1), Pixel(0x00FF0000)));
	g_camera = new Camera(Vector(0, 0, 10), createTransform(Vector(-1, 0, 0), Vector(0, 1, 0)));
}

void cleanup(HWND hWnd) {
	//Free up resources
	delete g_mesh;
	free(g_back_buff);
	free(g_zBuff);
	delete g_camera;
	DeleteObject(g_back_bmp);
	DeleteDC(g_backDC);
	ReleaseDC(hWnd, g_mainDC);
}

LRESULT CALLBACK WndProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam) {
	switch(msg) {
	case WM_CLOSE:
		DestroyWindow(hWnd);
		break;
	case WM_DESTROY:
		PostQuitMessage(0);
		break;
	case WM_COMMAND:
		switch(LOWORD(wParam)) {
		case ID_FILE_EXIT:
			PostMessage(hWnd, WM_CLOSE, 0, 0);
			break;
		}
		break;
	case WM_PAINT:
		PAINTSTRUCT ps;
		HDC hdc;
		
		hdc = BeginPaint(hWnd, &ps);
		
		render();
		
		EndPaint(hWnd, &ps);
		break;
	}
	
	return DefWindowProc(hWnd, msg, wParam, lParam);
}

void render() {
	BITMAPINFO bi;
	
	//Black background
	ZeroMemory(g_back_buff, sizeof(Pixel)*BUFF_WIDTH*BUFF_HEIGHT);
	FillMemory(g_zBuff, sizeof(unsigned long)*BUFF_WIDTH*BUFF_HEIGHT, 0xFF);
	
	//Draw the triangle
	g_camera->render(*g_mesh);
	
	//Copy the back buffer to the screen
	bi.bmiHeader.biSize = sizeof(bi.bmiHeader);
	bi.bmiHeader.biWidth = BUFF_WIDTH;
	bi.bmiHeader.biHeight = BUFF_HEIGHT;
	bi.bmiHeader.biPlanes = 1;
	bi.bmiHeader.biBitCount = 32;
	bi.bmiHeader.biCompression = BI_RGB;
	bi.bmiHeader.biSizeImage = 0;
	bi.bmiHeader.biClrUsed = 0;
	bi.bmiHeader.biClrImportant = 0;
	SetDIBits(g_backDC, g_back_bmp, 0, BUFF_HEIGHT, g_back_buff, &bi, DIB_RGB_COLORS);
	BitBlt(g_mainDC, 0, 0, BUFF_WIDTH, BUFF_HEIGHT, g_backDC, 0, 0, SRCCOPY);
}